%% 
% This is an example only.
% In this example, the script is parsing:
%   Example ground truth static data
%           & 
%   Example static data of the system to be evaluated: in this case, ZeroKey position data
% and computing the error between them.
% This script has a function that requires image processing and computer
% vision toolboxes. If they are not available, use Matlab online

close all
clear

%% Settings
% switch \ to become / if using Matlab Online
GTDataFileBase = 'TestData\gt_static_sameheading_point';
ZeroKeyDataFileBase = 'TestData\zk_static_sameheading_point';
StaticPoints = 40;
SamplesToAverage = 6;

%% Parse datasets

gt_x_all = [];
gt_y_all = [];
gt_z_all = [];

zk_x_all = [];
zk_y_all = [];
zk_z_all = [];

for i = 0:(StaticPoints-1)
    
    % Parse GT data file
    file_to_parse = sprintf('%s%d.txt', GTDataFileBase, i);
    [gt_nuc_unix, gt_meas_time, gt_x, gt_y, gt_z] = ParseGTDataFile(file_to_parse);
    % Grab the samples from the end of the dataset since the start likely
    % has some dynamic motion while the AGV settled
    gt_x_all = [gt_x_all; mean(gt_x(end-SamplesToAverage:end))];
    gt_y_all = [gt_y_all; mean(gt_y(end-SamplesToAverage:end))];
    gt_z_all = [gt_z_all; mean(gt_z(end-SamplesToAverage:end))];

    % Parse ZK data file
    file_to_parse = sprintf('%s%d.txt', ZeroKeyDataFileBase, i);
    [zk_nuc_unix,zk_us_seq,zk_ins_seq,zk_pos_x,zk_pos_y,zk_pos_z,zk_quat_w,zk_quat_x,zk_quat_y,zk_quat_z,zk_vel_x,zk_vel_y,zk_vel_z,zk_ins_flag,zk_us_flag,zk_relative_time,zk_tOffset] = ParseZeroKeyDataFile(file_to_parse);
    zk_x_all = [zk_x_all; mean(zk_pos_x(end-SamplesToAverage:end))];
    zk_y_all = [zk_y_all; mean(zk_pos_y(end-SamplesToAverage:end))];
    zk_z_all = [zk_z_all; mean(zk_pos_z(end-SamplesToAverage:end))];
end

%% Select outermost corner points for second comparison

% Compute distance to origin for all points
dTo = sqrt(zk_x_all .^ 2 + zk_y_all .^2);
[dTo, IA] = sort(dTo);

% Select closest and further points as 2 corner points
corner_points = [ IA(1) IA(end)];

points = [zk_x_all(corner_points), zk_y_all(corner_points)];
% Get unit vector from one corner to the other
uv = (points(1,:)-points(2,:))/norm(points(1,:)-points(2,:));
% Get orthogonal unit vector
ouv = null(uv'.');

% Find midpoint of line between two corner points
mp = uv * (norm(points(1,:)-points(2,:)) / 2) + points(2,:);

% Project orthogonal vector to find furtherst opposing corners
p1 = 50 * ouv(:,1)' + mp;
p2 = -50 * ouv(:,1)' + mp;

% Find closest anchor points to projected orthogonal endpoints
p1d = sqrt(sum((([zk_x_all, zk_y_all] - p1).^ 2)'));
[~, I] = min(p1d);
corner_points = [corner_points, I];

p2d = sqrt(sum((([zk_x_all, zk_y_all] - p2).^ 2)'));
[~, I] = min(p2d);
corner_points = [corner_points, I];

%% Solve transformation between all static points
% Requies image processing and computer vision toolboxes
% Run through Matlab online instead if you don't have it!
tform = estgeotform3d(...
    [gt_x_all, gt_y_all, gt_z_all], [zk_x_all, zk_y_all, zk_z_all],...
    "similarity",'MaxNumTrials',1e5, 'Confidence',99.9999);

% Transform GT points to ZK coordinate system
tf_gt_pos_static = [gt_x_all, gt_y_all, gt_z_all, ones(size(gt_x_all,1),1)];

tf_gt_pos_static = tf_gt_pos_static*tform.A';
tf_gt_pos_static = tf_gt_pos_static(:,1:3);

%% Solve transformation between corner points
tform_cp = estgeotform3d(...
    [gt_x_all(corner_points), gt_y_all(corner_points), gt_z_all(corner_points)],...
    [zk_x_all(corner_points), zk_y_all(corner_points), zk_z_all(corner_points)],...
    "similarity",'MaxNumTrials',1e5, 'Confidence',99.9999);
    
% Transform GT corner points to ZK coordinate system
tf_gt_cp_pos_static = [gt_x_all(corner_points), gt_y_all(corner_points),...
    gt_z_all(corner_points), ones(size(corner_points,2),1)];

tf_gt_cp_pos_static = tf_gt_cp_pos_static*tform_cp.A';
tf_gt_cp_pos_static = tf_gt_cp_pos_static(:,1:3);

%% Analysis of all points data

% Plot static points of both systems after transformation
figure;

plot3(zk_x_all,zk_y_all,zk_z_all,'*');
hold on
plot3(tf_gt_pos_static(:,1),tf_gt_pos_static(:,2),tf_gt_pos_static(:,3),'sq');
title('Transformed GT control points over ZK points');

error_mag = sqrt(sum((([zk_x_all, zk_y_all, zk_z_all] - tf_gt_pos_static(:,1:3)) .^ 2)'));
rmse_xyz = sqrt(mean(sum((([zk_x_all, zk_y_all, zk_z_all] - tf_gt_pos_static).^2)')));
rmse_xy = sqrt(mean(sum((([zk_x_all, zk_y_all] - tf_gt_pos_static(:,1:2)).^2)')));

disp(sprintf('Static position error using %d test points', StaticPoints));
T = table(rmse_xy, rmse_xyz, mean(error_mag), median(error_mag), std(error_mag), min(error_mag), max(error_mag),...
    'VariableNames', {'RMSE (XY)','RMSE (XYZ)','Mean','Median','STD','Min','Max'})

%% Analysis of corner points data
% Plot static points of both systems after transformation
figure;
plot3(zk_x_all(corner_points), zk_y_all(corner_points), zk_z_all(corner_points), '*');
hold on;
plot3(tf_gt_cp_pos_static(:,1),tf_gt_cp_pos_static(:,2),tf_gt_cp_pos_static(:,3),'sq');
title('Transformed corner points of GT control points over ZK points');

error_mag = sqrt(sum((([zk_x_all(corner_points), zk_y_all(corner_points), zk_z_all(corner_points)] - tf_gt_cp_pos_static(:,1:3)) .^ 2)'));
rmse_xyz = sqrt(mean(sum((([zk_x_all(corner_points), zk_y_all(corner_points), zk_z_all(corner_points)] - tf_gt_cp_pos_static).^2)')));
rmse_xy = sqrt(mean(sum((([zk_x_all(corner_points), zk_y_all(corner_points)] - tf_gt_cp_pos_static(:,1:2)).^2)')));

disp('Static position error using 4 corner test points');
T = table(rmse_xy, rmse_xyz, mean(error_mag), median(error_mag), std(error_mag), min(error_mag), max(error_mag),...
    'VariableNames', {'RMSE (XY)','RMSE (XYZ)','Mean','Median','STD','Min','Max'})


%% Parsing functions

% Parses a GT log file
function [gt_nuc_unix, gt_meas_time, gt_x, gt_y, gt_z] = ParseGTDataFile(filename)
    gt_nuc_unix = [];
    gt_meas_time = [];
    gt_x = [];
    gt_y = [];
    gt_z = [];
    
    gt_fid = fopen(filename);
    gt_meas_time = [];
    line = fgetl(gt_fid);
    while(line ~= -1)
        line = string(line);
        if ~isempty(line)
            str_split = strsplit(line);
            gt_nuc_unix = [gt_nuc_unix; str2double(str_split(2))];
            gt_meas_time = [gt_meas_time; str2double(str_split(4))];
            gt_x = [gt_x; str2double(str_split(6))];
            gt_y = [gt_y; str2double(str_split(7))];
            gt_z = [gt_z; str2double(str_split(8))];
        end
        line = fgetl(gt_fid);
    end
    fclose(gt_fid);
end

% Parses a ZeroKey data file
function [zk_nuc_unix,zk_us_seq,zk_ins_seq,zk_pos_x,zk_pos_y,zk_pos_z,zk_quat_w,zk_quat_x,zk_quat_y,zk_quat_z,zk_vel_x,zk_vel_y,zk_vel_z,zk_ins_flag,zk_us_flag,zk_relative_time,zk_tOffset] = ParseZeroKeyDataFile(filename)
    
    zk_nuc_unix = [];
    zk_us_seq = [];
    zk_ins_seq = [];
    zk_pos_x = [];
    zk_pos_y = [];
    zk_pos_z = [];
    zk_quat_w = [];
    zk_quat_x = [];
    zk_quat_y = [];
    zk_quat_z = [];
    zk_vel_x = [];
    zk_vel_y = [];
    zk_vel_z = [];
    zk_ins_flag = [];
    zk_us_flag = [];
    zk_relative_time = [];
    zk_tOffset = [];

    zk_fid = fopen(filename);

    line = fgetl(zk_fid);
    while (line ~= -1)
        line = string(line);
        if ~contains(line, "position")
            continue
        end
        %NUC_UNIX: 1716610767.9475427 : {"header": {"stamp": {"sec": 1716581386, "nanosec": 410900115}}, "unix_time": 1716581386.4109, "relative_time": "unavailable", "device_id": "F4:AC:E9:BF:A1:9F", "flag": "00", "UsSeq": 36093, "InsSeq": 4, "position": {"x": 1.31763, "y": 12.44981, "z": 0.44097}, "orientation": {"w": 0.73655, "x": 0.01408, "y": 0.0084, "z": -0.67619}, "velocity": {"x": 0.00322, "y": 0.0001, "z": 0.0005}, "pf": "43", "ts": "4962378512", "toffset": "16590"}
        pVals = regexp(line, '^.*NUC_UNIX: (?<NucUnix>[\d.]+) .*"UsSeq": (?<UsSeq>\d+),.*"InsSeq": (?<InsSeq>\d+),.*"position": {"x": (?<x>[-\d.Ee]+), "y": (?<y>[-\d.Ee]+), "z": (?<z>[-\d.Ee]+)}.*"orientation": {"w": (?<qw>[-\d.Ee]+), "x": (?<qx>[-\d.Ee]+), "y": (?<qy>[-\d.Ee]+), "z": (?<qz>[-\d.Ee]+)}.*"velocity": {"x": (?<vx>[-\d.Ee]+), "y": (?<vy>[-\d.Ee]+), "z": (?<vz>[-\d.Ee]+)}.*"pf": "(?<pf>[^"]*)".*"ts": "(?<ts>\d+)".*"toffset": "(?<toffset>\d+)"', 'names');
        if isempty(pVals)
            error(strcat('Error: Could not parse line:', line));
        end

        zk_nuc_unix = [zk_nuc_unix; str2double(pVals.NucUnix)];
        zk_us_seq = [zk_us_seq; str2double(pVals.UsSeq)];
        zk_ins_seq = [zk_ins_seq; str2double(pVals.InsSeq)];
        zk_pos_x = [zk_pos_x; str2double(pVals.x)];
        zk_pos_y = [zk_pos_y; str2double(pVals.y)];
        zk_pos_z = [zk_pos_z; str2double(pVals.z)];

        zk_quat_w = [zk_quat_w; str2double(pVals.qw)];
        zk_quat_x = [zk_quat_x; str2double(pVals.qx)];
        zk_quat_y = [zk_quat_y; str2double(pVals.qy)];
        zk_quat_z = [zk_quat_z; str2double(pVals.qz)];

        zk_vel_x = [zk_vel_x; str2double(pVals.vx)];
        zk_vel_y = [zk_vel_y; str2double(pVals.vy)];
        zk_vel_z = [zk_vel_z; str2double(pVals.vz)];

        zk_ins_flag = [zk_ins_flag; str2double(pVals.pf{1}(1))];
        zk_us_flag = [zk_us_flag; str2double(pVals.pf{1}(2))];
        zk_relative_time = [zk_relative_time; str2double(pVals.ts)];
        zk_tOffset = [zk_tOffset; str2double(pVals.toffset)];

        line = fgetl(zk_fid);
    end
    fclose(zk_fid);
end